<?php

declare(strict_types=1);

namespace Brick\Math\Tests;

use Brick\Math\BigDecimal;
use Brick\Math\BigInteger;
use Brick\Math\BigRational;
use PHPUnit\Framework\TestCase;

use function str_ends_with;

/**
 * Base class for math tests.
 */
abstract class AbstractTestCase extends TestCase
{
    /**
     * @param string     $expected The expected value, as a string.
     * @param BigInteger $actual   The BigInteger instance to test.
     */
    final protected static function assertBigIntegerEquals(string $expected, BigInteger $actual): void
    {
        self::assertSame($expected, (string) $actual);
    }

    /**
     * @param string     $expected The expected string representation.
     * @param BigDecimal $actual   The BigDecimal instance to test.
     */
    final protected static function assertBigDecimalEquals(string $expected, BigDecimal $actual): void
    {
        self::assertSame($expected, (string) $actual);
    }

    /**
     * @param string      $expected The expected string representation.
     * @param BigRational $actual   The BigRational instance to test.
     */
    final protected static function assertBigRationalEquals(string $expected, BigRational $actual): void
    {
        self::assertSame($expected, (string) $actual);
    }

    /**
     * @param string     $unscaledValue The expected unscaled value, as a string.
     * @param int        $scale         The expected scale.
     * @param BigDecimal $actual        The BigDecimal instance to test.
     */
    final protected static function assertBigDecimalInternalValues(string $unscaledValue, int $scale, BigDecimal $actual): void
    {
        self::assertSame($unscaledValue, (string) $actual->getUnscaledValue(), 'Unscaled value mismatch');
        self::assertSame($scale, $actual->getScale(), 'Scale mismatch');
    }

    /**
     * @param string      $numerator   The expected numerator, as a string.
     * @param string      $denominator The expected denominator, as a string.
     * @param BigRational $actual      The BigRational instance to test.
     */
    final protected static function assertBigRationalInternalValues(string $numerator, string $denominator, BigRational $actual): void
    {
        self::assertSame($numerator, (string) $actual->getNumerator(), 'Numerator mismatch');
        self::assertSame($denominator, (string) $actual->getDenominator(), 'Denominator mismatch');
    }

    final protected static function isException(string $name): bool
    {
        return str_ends_with($name, 'Exception');
    }
}
